<?php
/**
* @author Amasty Team
* @copyright Copyright (c) 2022 Amasty (https://www.amasty.com)
* @package Product Feed for Magento 2
*/

namespace Amasty\Feed\Model\Export\RowCustomizer;

use Amasty\Base\Model\Serializer;
use Amasty\Feed\Model\Config\Source\Path;
use Amasty\Feed\Model\Export\Product as ExportProduct;
use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\CatalogImportExport\Model\Import\Product as ImportProduct;
use Magento\CatalogImportExport\Model\Export\RowCustomizerInterface;
use Magento\CatalogUrlRewrite\Model\ProductUrlRewriteGenerator;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\ImportExport\Model\Import;
use Magento\Store\Model\ScopeInterface;
use Magento\Store\Model\StoreManagerInterface;

class Url implements RowCustomizerInterface
{
    /**
     * @var StoreManagerInterface
     */
    protected $storeManager;

    /**
     * @var array
     */
    protected $urlRewrites;

    /**
     * @var \Magento\Framework\Url
     */
    protected $url;

    /**
     * @var int
     */
    protected $storeId;

    /**
     * @var array
     */
    protected $rowCategories;

    /**
     * @var ExportProduct
     */
    protected $export;

    /**
     * @var ScopeConfigInterface
     */
    private $scopeConfig;

    /**
     * @var Serializer
     */
    private $serializer;

    /**
     * @var ProductRepositoryInterface
     */
    private $productRepository;

    public function __construct(
        StoreManagerInterface $storeManager,
        ExportProduct $export,
        \Magento\Framework\Url $url, //always get frontend url
        ScopeConfigInterface $scopeConfig,
        Serializer $serializer,
        ProductRepositoryInterface $productRepository
    ) {
        $this->url = $url;
        $this->export = $export;
        $this->storeManager = $storeManager;
        $this->scopeConfig = $scopeConfig;
        $this->serializer = $serializer;
        $this->productRepository = $productRepository;
    }

    /**
     * @inheritdoc
     */
    public function prepareData($collection, $productIds)
    {
        if ($this->export->hasAttributes(ExportProduct::PREFIX_URL_ATTRIBUTE)) {
            $this->storeId = $collection->getStoreId();
            $select = $collection->getConnection()
                ->select()
                ->from(['u' => $collection->getTable('url_rewrite')], ['u.entity_id', 'u.request_path', 'u.metadata'])
                ->where('u.store_id = ?', $this->storeId)
                ->where('u.is_autogenerated = 1')
                ->where('u.entity_type = ?', ProductUrlRewriteGenerator::ENTITY_TYPE)
                ->where('u.entity_id IN(?)', $productIds);

            foreach ($collection->getConnection()->fetchAll($select) as $row) {
                $metadata = $this->serializer->unserialize($row['metadata']);
                $categoryId = is_array($metadata) && isset($metadata['category_id'])
                    ? $metadata['category_id']
                    : null;

                if (!isset($row['entity_id'])) {
                    $this->urlRewrites[$row['entity_id']] = [];
                }

                $this->urlRewrites[$row['entity_id']][(int)$categoryId] = $row['request_path'];
            }

            $multiRowData = $this->export->getMultiRowData();
            $this->rowCategories = $multiRowData['rowCategories'];
        }
    }

    /**
     * @inheritdoc
     */
    public function addHeaderColumns($columns)
    {
        return $columns;
    }

    /**
     * @inheritdoc
     */
    public function addData($dataRow, $productId)
    {
        $customData = &$dataRow['amasty_custom_data'];
        //if the production mode + config cache is enabled.
        //Store manager returns current website url, instead of one have been set.
        //taking url directly from URL model.
        $this->url->setScope($this->storeManager->getStore($this->storeId));

        if ($this->urlRewrites && isset($this->urlRewrites[$productId])) {
            $urlRewrites = $this->urlRewrites[$productId];
            $pathMode = $this->scopeConfig->getValue(
                'amasty_feed/general/category_path',
                ScopeInterface::SCOPE_STORE
            );

            if (count($urlRewrites) > 1 && $pathMode != Path::USE_DEFAULT) {
                $categoryRewrites = array_slice($urlRewrites, 1);

                if ($pathMode == Path::USE_SHORTEST) {
                    uasort(
                        $categoryRewrites,
                        function ($a, $b) {
                            return strlen($a) > strlen($b) ? 1 : -1;
                        }
                    );
                } else {
                    uasort(
                        $categoryRewrites,
                        function ($a, $b) {
                            return strlen($a) < strlen($b) ? 1 : -1;
                        }
                    );
                }

                $urlWithCategory = reset($categoryRewrites);
            } else {
                $categories = $this->rowCategories[$productId] ?? [];
                $lastCategoryId = count($categories) > 0 ? end($categories) : null;
                $urlWithCategory = $urlRewrites[$lastCategoryId] ?? end($urlRewrites);
            }

            $routeParamsShort = [
                '_direct' => $urlRewrites[0] ?? end($urlRewrites),
                '_nosid' => true,
                '_query' => array_merge((array)$this->export->getUtmParams(), ['___store' => null]),
                '_scope_to_url' => true, //as in  \Magento\Store\Model\Store::getUrl()
                '_scope' => $this->url->getScope(),
            ];
            $routeParamsWithCategory = [
                '_direct' => $urlWithCategory,
                '_nosid' => true,
                '_query' => array_merge((array)$this->export->getUtmParams(), ['___store' => null]),
                '_scope_to_url' => true, //as in  \Magento\Store\Model\Store::getUrl()
                '_scope' => $this->url->getScope(),
            ];
            $customData[ExportProduct::PREFIX_URL_ATTRIBUTE] = [
                'short' => $this->url->getUrl('', $routeParamsShort),
                'with_category' => $this->url->getUrl('', $routeParamsWithCategory),
            ];
        } else {
            $lastCategoryId = $dataRow['amasty_custom_data'][ExportProduct::PREFIX_CATEGORY_ID_ATTRIBUTE] ?? null;
            $routeParamsShort = [
                '_nosid' => true,
                '_query' => array_merge((array)$this->export->getUtmParams(), ['___store' => null]),
                '_scope_to_url' => true,
                'id' => $productId,
                's' => $dataRow['url_key'] ?? '',
                '_scope' => $this->url->getScope(),
            ];
            $routeParamsWithCategory = array_merge($routeParamsShort, ['category' => $lastCategoryId]);
            $customData[ExportProduct::PREFIX_URL_ATTRIBUTE] = [
                'short' => $this->url->getUrl('catalog/product/view', $routeParamsShort),
                'with_category' => $this->url->getUrl('catalog/product/view', $routeParamsWithCategory)
            ];
        }

        $customData[ExportProduct::PREFIX_URL_ATTRIBUTE]['configurable'] = $this->getConfigurableUrl(
            $customData,
            $dataRow['sku']
        );

        return $dataRow;
    }

    /**
     * @inheritdoc
     */
    public function getAdditionalRowsCount($additionalRowsCount, $productId)
    {
        return $additionalRowsCount;
    }

    private function getConfigurableUrl(array $customData, string $sku): string
    {
        $parentConfiguration = $customData['parent_data']['configurable_variations'][$sku] ?? [];
        $configurableUrl = $customData['parent_data']['amasty_custom_data']['url']['short'] ?? '';
        if ($parentConfiguration && $configurableUrl) {
            $config = [];

            foreach ($parentConfiguration as $attrId => $attrValueId) {
                $config[] = $attrId . '=' . $attrValueId;
            }
            if ($config) {
                $configurableUrl = rtrim($configurableUrl, '/') . '#' . implode('&', $config);
            }
        } else {
            $configurableUrl = $customData[ExportProduct::PREFIX_URL_ATTRIBUTE]['short'];
        }

        return $configurableUrl;
    }
}
